<?php

namespace App\Actions\Fortify;

use App\Models\User;
use App\Mail\UserWelcomeMail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Laravel\Fortify\Contracts\CreatesNewUsers;
use Laravel\Jetstream\Jetstream;

class CreateNewUser implements CreatesNewUsers
{
    use PasswordValidationRules;

    /**
     * Validate and create a newly registered user.
     */
    public function create(array $input): User
    {
        Validator::make($input, [
            'first_name' => ['required', 'string', 'max:255'],
            'middle_name' => ['nullable', 'string', 'max:255'],
            'surname' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'phone_number' => ['required', 'string', 'max:20'],
            'organization' => ['required', 'string', 'max:255'],
            'password' => $this->passwordRules(),
            'terms' => Jetstream::hasTermsAndPrivacyPolicyFeature() ? ['accepted', 'required'] : '',
        ])->validate();

        $user = User::create([
            'first_name' => $input['first_name'],
            'middle_name' => $input['middle_name'] ?? null,
            'surname' => $input['surname'],
            'email' => $input['email'],
            'phone_number' => $input['phone_number'],
            'role' => 'user',
            'organization' => $input['organization'],
            'password' => Hash::make($input['password']),
        ]);

        try {
            Mail::mailer('noreply')->to($user->email)->send(new UserWelcomeMail($user));
        } catch (\Exception $e) {
            // Log the error but continue registration
            logger()->error('Failed to send welcome email: ' . $e->getMessage());
        }

        return $user;
    }
}